#!/bin/bash
echo 'Checking environment...'

# Function to check Python version
check_python_version() {
        local py_command=$1
        local version=$("$py_command" -V 2>&1 | awk '{print $2}')
        local U_V1=$(echo "$version" | awk -F '.' '{print $1}')
        local U_V2=$(echo "$version" | awk -F '.' '{print $2}')

        if [ "$U_V1" -eq 3 ] && [ "$U_V2" -eq 10 ]; then
                echo "$py_command (version $version) is suitable."
                PYTHON_CMD=$py_command
                return 0
        else
                echo "$py_command version is not 3.10.x (found $version)" >&2
                return 1
        fi
}

# Install system dependencies
echo "Installing system dependencies..."
if command -v apt-get &>/dev/null; then
        # Update package list first
        sudo apt-get update
        # Install required packages
        sudo apt-get install -y --no-install-recommends \
                libtinfo5 \
                libgl1-mesa-glx \
                >/dev/null

        echo "System dependencies installed successfully."
else
        echo "apt-get not found. Skipping system dependencies installation." >&2
        echo "Please ensure libtinfo5 is installed manually." >&2
fi

# Check for python3, then fallback to python if necessary
PYTHON_CMD=""
if command -v python3 &>/dev/null; then
        check_python_version python3 || true
fi

if [ -z "$PYTHON_CMD" ] && command -v python &>/dev/null; then
        check_python_version python || true
fi

# Exit if neither python3 nor python meet the version requirement
if [ -z "$PYTHON_CMD" ]; then
        echo "Neither Python 3.10 nor Python 3.10-compatible version found!" >&2
        exit 1
fi

# Check if pip3 or pip is available
if command -v pip3 &>/dev/null; then
        PIP_CMD=pip3
elif command -v pip &>/dev/null; then
        PIP_CMD=pip
else
        echo "Neither pip3 nor pip is installed!" >&2
        exit 1
fi

# Find the latest CixBuilder wheel file and install it
target_file=$(ls CixBuilder-*-cp310-none-linux_x86_64.whl 2>/dev/null | sort -V | tail -n 1)
if [ -n "$target_file" ]; then
        echo "Installing $target_file using $PIP_CMD"
        "$PIP_CMD" install "$target_file"
else
        echo "No CixBuilder wheel file found in the current directory." >&2
        exit 1
fi
